<?php
/**
 * Helpers.
 *
 * @package Page Builder Framework
 */

defined( 'ABSPATH' ) || die( "Can't access directly" );

/**
 * Custom strpos array helper function.
 *
 * @param string  $haystack The haystack.
 * @param array   $needles The needles.
 * @param integer $offset The offset.
 *
 * @return boolean.
 */
function wpbf_strposa( $haystack, $needles, $offset = 0 ) {

	if ( ! is_array( $needles ) ) {
		$needles = array( $needles );
	}

	foreach ( $needles as $needle ) {
		if ( strpos( $haystack, $needle, $offset ) !== false ) {
			return true;
		}
	}

	return false;

}

/**
 * Pingback head link.
 *
 * Add Pingback head link if we're on singular and pings are open.
 */
function wpbf_pingback_header() {

	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="' . esc_url( get_bloginfo( 'pingback_url' ) ) . '">';
	}

}
add_action( 'wp_head', 'wpbf_pingback_header' );

/**
 * Schema markup (body).
 */
function wpbf_body_schema_markup() {

	// Blog variable.
	$is_blog = ( is_home() || is_date() || is_category() || is_author() || is_tag() || is_attachment() || is_singular( 'post' ) ) ? true : false;

	// Default itemtype.
	$itemtype = 'WebPage';

	// Define itemtype for blog pages, otherwise use WebPage.
	$itemtype = ( $is_blog ) ? 'Blog' : $itemtype;

	// Define itemtype for search results, otherwise use WebPage.
	$itemtype = ( is_search() ) ? 'SearchResultsPage' : $itemtype;

	// Make result filterable.
	$result = apply_filters( 'wpbf_body_itemtype', $itemtype );

	// Output.
	echo 'itemscope="itemscope" itemtype="https://schema.org/' . esc_html( $result ) . '"';

}

/**
 * Schema markup (archive).
 */
function wpbf_archive_schema_markup() {

	// Default itemtype.
	$itemtype = 'CreativeWork';

	// Make result filterable.
	$result = apply_filters( 'wpbf_archive_itemtype', $itemtype );

	// Output.
	echo 'itemscope="itemscope" itemtype="https://schema.org/' . esc_html( $result ) . '"';

}

/**
 * Schema markup (single).
 */
function wpbf_single_schema_markup() {

	// Default itemtype.
	$itemtype = 'CreativeWork';

	// Make result filterable.
	$result = apply_filters( 'wpbf_single_itemtype', $itemtype );

	// Output.
	echo 'itemscope="itemscope" itemtype="https://schema.org/' . esc_html( $result ) . '"';

}

/**
 * Custom comments callback.
 *
 * @param Object  $comment WP_Comment.
 * @param array   $args The arguments.
 * @param integer $depth The depth.
 */
function wpbf_comments( $comment, $args, $depth ) {

	// Pingbacks & Trackbacks.
	if ( 'pingback' === $comment->comment_type || 'trackback' === $comment->comment_type ) { ?>

		<li id="comment-<?php comment_ID(); ?>">

			<article <?php comment_class(); ?>>

				<?php _e( 'Pingback:', 'page-builder-framework' ); ?>

				<?php comment_author_link(); ?>

				<?php edit_comment_link(); ?>

			</article>

		<?php // We're not closing the li tag right here. WordPress does it for us. ?>

	<?php } else { ?>

		<li id="comment-<?php comment_ID(); ?>">

			<article <?php comment_class(); ?> itemscope="itemscope" itemtype="https://schema.org/Comment">

				<footer class="comment-meta">

					<?php echo get_avatar( $comment, 120 ); ?>

					<div class="comment-author-info">

						<span class="comment-author author vcard" itemprop="author" itemscope itemtype="https://schema.org/Person">

						<?php

						if ( $comment->user_id ) {
							$user = get_userdata( $comment->user_id );
							printf( '<cite itemprop="name" class="fn">%s</cite>', esc_html( $user->display_name ) );
						} else {
							printf( '<cite itemprop="name" class="fn">%s</cite>', get_comment_author_link() );
						}

						?>

						</span>

						<?php edit_comment_link( __( 'Edit', 'page-builder-framework' ) ); ?>

						<?php
						if ( wpbf_svg_enabled() ) {
							$date_icon = wpbf_svg( 'clock' );
						} else {
							$date_icon = '<i class="wpbff wpbff-clock"></i>';
						}
						?>

						<time class="comment-time" datetime="<?php comment_time( 'c' ); ?>" itemprop="datePublished">

							<a href="<?php echo esc_url( get_comment_link( $comment->comment_ID ) ); ?>"><?php echo $date_icon; ?> <?php comment_time( __( 'F d, Y', 'page-builder-framework' ) ); ?></a>

						</time>

					</div>

				</footer>

				<?php if ( '0' == $comment->comment_approved ) : ?>

				<div class="wpbf-notice wpbf-notice-warning">
					<?php _e( 'Your comment has yet to be approved.', 'page-builder-framework' ); ?>
				</div>

				<?php endif; ?>

				<div class="comment-content" itemprop="text">

					<?php comment_text(); ?>

				</div>

				<?php
				comment_reply_link(
					array_merge(
						$args,
						array(
							'depth'     => $depth,
							'max_depth' => $args['max_depth'],
						)
					)
				);
				?>

			</article>

		<?php
		// We're not closing the li tag right here. WordPress does it for us.
	}

}


if ( ! function_exists( 'wpbf_svg_enabled' ) ) {

	/**
	 * Helper function to check if SVG's are enabled.
	 */
	function wpbf_svg_enabled() {

		if ( apply_filters( 'wpbf_svg', false ) ) {
			return true;
		}

		return false;

	}

}

if ( ! function_exists( 'wpbf_svg' ) ) {

	/**
	 * Function to call SVG icons.
	 *
	 * @param string $icon The icon to call.
	 */
	function wpbf_svg( $icon ) {

		$output = '';

		if ( 'arrow-down' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M15.35 18.544l-10.166-10.166c-0.36-0.36-0.943-0.36-1.302 0l-1.23 1.23c-0.36 0.36-0.36 0.943 0 1.302l12.698 12.698c0.36 0.36 0.943 0.36 1.302 0l12.698-12.698c0.36-0.36 0.36-0.943 0-1.302l-1.23-1.23c-0.36-0.36-0.943-0.36-1.302 0l-10.167 10.166c-0.36 0.36-0.943 0.36-1.302 0z"></path>
						</svg>';
		}

		if ( 'arrow-left' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M13.449 15.341l10.167-10.166c0.36-0.36 0.36-0.943 0-1.302l-1.23-1.23c-0.36-0.36-0.943-0.36-1.302 0l-12.698 12.698c-0.36 0.36-0.36 0.943 0 1.302l12.698 12.698c0.36 0.36 0.943 0.36 1.302 0l1.23-1.23c0.36-0.36 0.36-0.943 0-1.302l-10.167-10.166c-0.36-0.36-0.36-0.943 0-1.302z"></path>
						</svg>';
		}

		if ( 'arrow-right' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M18.55 16.643l-10.159 10.159c-0.359 0.359-0.359 0.942 0 1.301l1.229 1.229c0.359 0.359 0.942 0.359 1.301 0l12.69-12.689c0.359-0.359 0.359-0.942 0-1.301l-12.69-12.69c-0.359-0.359-0.942-0.359-1.301 0l-1.229 1.229c-0.359 0.359-0.359 0.942 0 1.301l10.159 10.159c0.359 0.359 0.359 0.942 0 1.301z"></path>
						</svg>';
		}

		if ( 'arrow-up' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M16.652 13.443l10.159 10.159c0.359 0.359 0.942 0.359 1.301 0l1.229-1.229c0.359-0.359 0.359-0.942 0-1.301l-12.689-12.69c-0.359-0.359-0.942-0.359-1.301 0l-12.69 12.69c-0.359 0.359-0.359 0.942 0 1.301l1.229 1.229c0.359 0.359 0.942 0.359 1.301 0l10.159-10.159c0.359-0.359 0.942-0.359 1.301 0z"></path>
						</svg>';
		}

		if ( 'bag' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M12.146 7.546c0.046-2.094 1.758-3.778 3.864-3.778s3.817 1.684 3.864 3.778h-7.727zM16 0.846c-3.717 0-6.735 2.993-6.777 6.7h-2.397c-0.827 0-1.524 0.617-1.624 1.438l-2.473 20.308c-0.119 0.974 0.642 1.833 1.624 1.833h23.292c0.982 0 1.742-0.859 1.624-1.833l-2.473-20.308c-0.1-0.821-0.797-1.438-1.624-1.438h-2.397c-0.042-3.707-3.060-6.7-6.777-6.7zM23.995 10.513l2.148 17.646h-20.286l2.148-17.646h15.99z"></path>
						</svg>';
		}

		if ( 'basket' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M24.343,10.641l-6.128,-6.128c-0.611,-0.611 -0.611,-1.601 0,-2.212c0.611,-0.611 1.602,-0.611 2.213,-0l8.341,8.34l2.224,0c0.6,0 1.055,0.542 0.952,1.134l-3.165,17.589c-0.081,0.463 -0.482,0.8 -0.951,0.8l-23.655,-0c-0.47,-0 -0.871,-0.337 -0.952,-0.8l-3.164,-17.589c-0.104,-0.592 0.351,-1.134 0.952,-1.134l2.227,0l8.341,-8.34c0.611,-0.611 1.601,-0.611 2.212,-0c0.611,0.611 0.611,1.601 0,2.212l-6.128,6.128l16.681,0Zm-17.129,15.853l17.574,-0c0.469,-0 0.871,-0.338 0.952,-0.8l1.785,-10.249c0.103,-0.591 -0.352,-1.133 -0.952,-1.133l-21.143,-0c-0.601,-0 -1.056,0.542 -0.953,1.133l1.786,10.249c0.081,0.463 0.482,0.8 0.951,0.8Zm14.413,-2.36c-0.864,-0 -1.564,-0.701 -1.564,-1.565l-0,-4.332c-0,-0.864 0.7,-1.565 1.564,-1.565c0.864,0 1.565,0.701 1.565,1.565l-0,4.332c-0,0.864 -0.701,1.565 -1.565,1.565Zm-11.252,-0c-0.864,-0 -1.564,-0.701 -1.564,-1.565l-0,-4.332c-0,-0.864 0.7,-1.565 1.564,-1.565c0.864,0 1.565,0.701 1.565,1.565l-0,4.332c-0,0.864 -0.701,1.565 -1.565,1.565Zm5.626,-0c-0.864,-0 -1.564,-0.701 -1.564,-1.565l-0,-4.332c-0,-0.864 0.7,-1.565 1.564,-1.565c0.864,0 1.565,0.701 1.565,1.565l-0,4.332c-0,0.864 -0.701,1.565 -1.565,1.565Z"></path>
						</svg>';
		}

		if ( 'behance' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M29.627 0.017c1.303 0 2.365 1.032 2.365 2.305v27.359c0 1.273-1.062 2.307-2.365 2.307h-27.247c-1.301 0-2.358-1.034-2.358-2.307v-27.359c0-1.273 1.057-2.305 2.358-2.305h27.247zM10.838 23.319c0.742 0 1.43-0.066 2.066-0.198 0.634-0.131 1.185-0.377 1.651-0.735 0.413-0.31 0.758-0.694 1.035-1.15 0.436-0.688 0.65-1.467 0.65-2.335 0-0.84-0.19-1.554-0.57-2.143-0.378-0.591-0.942-1.020-1.687-1.293 0.492-0.25 0.863-0.528 1.114-0.832 0.452-0.544 0.676-1.26 0.676-2.154 0-0.867-0.222-1.613-0.669-2.234-0.746-1.013-2.008-1.53-3.79-1.559h-7.067v14.633h6.59zM27.763 18.394c0.013-0.908-0.019-1.603-0.090-2.081-0.129-0.822-0.406-1.544-0.836-2.168-0.476-0.709-1.079-1.224-1.809-1.552-0.733-0.327-1.554-0.49-2.465-0.49-1.537 0-2.784 0.483-3.75 1.454-0.962 0.969-1.442 2.361-1.442 4.179 0 1.939 0.534 3.337 1.598 4.199 1.067 0.859 2.297 1.29 3.692 1.29 1.689 0 3.004-0.511 3.942-1.53 0.605-0.641 0.942-1.272 1.015-1.895h-2.797c-0.162 0.306-0.35 0.548-0.563 0.719-0.391 0.324-0.898 0.481-1.521 0.481-0.59 0-1.095-0.129-1.511-0.393-0.69-0.422-1.053-1.158-1.095-2.212h7.631zM10.764 16.878c0.627 0.006 1.116 0.088 1.463 0.248 0.62 0.282 0.931 0.807 0.931 1.567 0 0.901-0.321 1.508-0.961 1.827-0.355 0.174-0.847 0.257-1.482 0.257h-3.092v-3.9h3.141zM24.919 16.586c-0.051-0.728-0.294-1.28-0.732-1.656-0.434-0.378-0.977-0.568-1.624-0.568-0.705 0-1.249 0.203-1.635 0.601-0.389 0.398-0.631 0.938-0.73 1.623h4.722zM10.346 11.227c0.7 0 1.277 0.078 1.727 0.229 0.524 0.218 0.786 0.669 0.786 1.35 0 0.616-0.197 1.045-0.594 1.287s-0.911 0.363-1.547 0.363h-3.096v-3.228h2.723zM25.486 10.757v-1.464h-5.893v1.464h5.893z"></path>
						</svg>';
		}

		if ( 'caret-down' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M2.977 11.592l11.623 11.623c0.774 0.774 2.028 0.774 2.802 0l11.623-11.623c1.248-1.248 0.364-3.382-1.401-3.382h-23.246c-1.765 0-2.649 2.134-1.401 3.382z"></path>
						</svg>';
		}

		if ( 'caret-left' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M20.41 2.98l-11.623 11.623c-0.774 0.774-0.774 2.028 0 2.802l11.623 11.623c1.248 1.248 3.382 0.364 3.382-1.401v-23.246c0-1.765-2.134-2.649-3.382-1.401z"></path>
						</svg>';
		}

		if ( 'caret-right' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M11.593 29.018l11.615-11.615c0.773-0.773 0.773-2.027 0-2.8l-11.615-11.615c-1.247-1.247-3.379-0.364-3.379 1.4v23.229c0 1.764 2.132 2.647 3.379 1.4z"></path>
						</svg>';
		}

		if ( 'caret-up' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M29.015 20.379l-11.615-11.615c-0.773-0.773-2.027-0.773-2.8 0l-11.615 11.615c-1.247 1.247-0.364 3.379 1.4 3.379h23.229c1.764 0 2.647-2.132 1.4-3.379z"></path>
						</svg>';
		}

		if ( 'cart' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M10.191 16.429h15.95c0.31 0 0.573-0.221 0.617-0.519l1.162-7.92c0.053-0.364-0.238-0.689-0.617-0.689h-18.273c-0.379 0-0.67 0.325-0.617 0.689l1.162 7.92c0.044 0.298 0.307 0.519 0.617 0.519zM29.194 19.837h-22.056c-0.31 0-0.573-0.221-0.617-0.519l-2.161-14.737c-0.053-0.364 0.238-0.689 0.617-0.689h26.379c0.379 0 0.67 0.325 0.617 0.689l-2.161 14.737c-0.044 0.298-0.307 0.519-0.617 0.519z"></path>
							<path d="M26.21 23.631l-2.34 2.343h-11.9l-2.824-2.665-2.412 0.727-2.784-19.264c-0.047-0.319-0.32-0.556-0.643-0.556h-2.634c-0.359 0-0.65-0.291-0.65-0.65v-2.218c0-0.359 0.291-0.65 0.65-0.65h5.674c0.323 0 0.596 0.237 0.643 0.556l2.986 20.647c0.047 0.319 0.32 0.556 0.643 0.556h16.347c0.359 0 0.65 0.291 0.65 0.65l1.128 1.403-2.534-0.878z"></path>
							<path d="M26.454 24.857c-1.070 0-1.94 0.87-1.94 1.94s0.87 1.94 1.94 1.94c1.070 0 1.94-0.87 1.94-1.94s-0.87-1.94-1.94-1.94zM26.454 31.308c-2.487 0-4.51-2.023-4.51-4.51s2.023-4.51 4.51-4.51c2.487 0 4.51 2.023 4.51 4.51s-2.023 4.51-4.51 4.51z"></path>
							<path d="M8.635 24.857c-1.070 0-1.94 0.87-1.94 1.94s0.87 1.94 1.94 1.94c1.070 0 1.94-0.87 1.94-1.94s-0.87-1.94-1.94-1.94zM8.635 31.308c-2.487 0-4.51-2.023-4.51-4.51s2.023-4.51 4.51-4.51c2.487 0 4.51 2.023 4.51 4.51s-2.023 4.51-4.51 4.51z"></path>
						</svg>';
		}

		if ( 'clock' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M16 28.38c-6.729 0-12.388-5.658-12.388-12.388s5.658-12.388 12.388-12.388c6.729 0 12.388 5.658 12.388 12.388s-5.658 12.388-12.388 12.388zM16 0.018c-8.822 0-15.974 7.152-15.974 15.974s7.152 15.974 15.974 15.974c8.822 0 15.974-7.152 15.974-15.974s-7.152-15.974-15.974-15.974z"></path>
							<path d="M17.435 13.771v-6.475c0-0.508-0.412-0.92-0.92-0.92h-1.738c-0.508 0-0.92 0.412-0.92 0.92v10.053c0 0.508 0.412 0.92 0.92 0.92h8.249c0.508 0 0.92-0.412 0.92-0.92v-1.738c0-0.508-0.412-0.92-0.92-0.92h-4.671c-0.508 0-0.92-0.412-0.92-0.92z"></path>
						</svg>';
		}

		if ( 'dribbble' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M15.999 0.018c-8.822 0-15.981 7.159-15.981 15.981s7.159 15.981 15.981 15.981c8.805 0 15.981-7.159 15.981-15.981s-7.176-15.981-15.981-15.981zM26.554 7.384c1.907 2.323 3.051 5.287 3.085 8.493-0.451-0.087-4.957-1.005-9.498-0.433-0.104-0.225-0.191-0.468-0.295-0.711-0.277-0.658-0.589-1.334-0.901-1.976 5.027-2.045 7.315-4.992 7.609-5.373zM15.999 2.375c3.467 0 6.639 1.3 9.048 3.432-0.243 0.346-2.305 3.103-7.159 4.923-2.236-4.108-4.714-7.471-5.096-7.991 1.022-0.243 2.097-0.364 3.206-0.364zM10.192 3.658c0.364 0.485 2.791 3.865 5.061 7.887-6.378 1.698-12.012 1.664-12.618 1.664 0.884-4.229 3.744-7.748 7.557-9.551zM2.34 16.016v-0.416c0.589 0.017 7.211 0.104 14.022-1.941 0.399 0.763 0.763 1.543 1.109 2.323-0.173 0.052-0.364 0.104-0.537 0.156-7.037 2.271-10.781 8.476-11.093 8.996-2.166-2.409-3.501-5.616-3.501-9.117zM15.999 29.657c-3.155 0-6.066-1.075-8.372-2.877 0.243-0.503 3.016-5.841 10.712-8.528 0.035-0.017 0.052-0.017 0.087-0.035 1.924 4.975 2.704 9.152 2.912 10.348-1.647 0.711-3.449 1.092-5.339 1.092zM23.608 27.317c-0.139-0.832-0.866-4.819-2.652-9.724 4.281-0.676 8.025 0.433 8.493 0.589-0.589 3.796-2.773 7.072-5.841 9.134z"></path>
						</svg>';
		}

		if ( 'email' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M1.024 4.013c0.011-0 0.023-0 0.034-0h29.876c0.011 0 0.023 0 0.034 0 0.561 0.015 1.009 0.361 1.009 0.785v22.402c0 0.424-0.448 0.77-1.009 0.785-0.011 0-29.933 0-29.944 0-0.561-0.015-1.009-0.361-1.009-0.785v-22.402c0-0.424 0.448-0.77 1.009-0.785zM3.593 13.783v10.624h24.806v-10.624l-12.145 7.012c-0.081 0.047-0.176 0.060-0.277 0.044-0.090 0.009-0.173-0.006-0.246-0.048l-12.138-7.008zM3.593 7.59v2.061l12.403 7.161 12.403-7.161v-2.061h-24.806z"></path>
						</svg>';
		}

		if ( 'facebook' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M30.218 0.021h-28.436c-0.975 0-1.764 0.79-1.764 1.764v28.436c0 0.974 0.789 1.764 1.764 1.764h15.309v-12.378h-4.166v-4.824h4.166v-3.558c0-4.129 2.521-6.377 6.205-6.377 1.764 0 3.28 0.131 3.722 0.19v4.315l-2.555 0.001c-2.003 0-2.391 0.952-2.391 2.349v3.080h4.777l-0.622 4.824h-4.155v12.378h8.145c0.974 0 1.764-0.79 1.764-1.764v-28.436c0-0.974-0.79-1.764-1.764-1.764z"></path>
						</svg>';
		}

		if ( 'github' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M16.001 0.021c-8.824 0-15.981 7.156-15.981 15.981 0 7.061 4.579 13.051 10.93 15.165 0.798 0.148 1.056-0.348 1.056-0.768v-2.975c-4.445 0.967-5.371-1.886-5.371-1.886-0.727-1.847-1.775-2.339-1.775-2.339-1.45-0.992 0.111-0.971 0.111-0.971 1.605 0.112 2.449 1.647 2.449 1.647 1.425 2.442 3.738 1.737 4.65 1.328 0.143-1.032 0.557-1.738 1.015-2.136-3.549-0.406-7.281-1.777-7.281-7.899 0-1.746 0.625-3.171 1.646-4.29-0.165-0.404-0.713-2.030 0.156-4.23 0 0 1.342-0.429 4.396 1.638 1.274-0.354 2.641-0.531 3.999-0.538 1.358 0.007 2.726 0.184 4.003 0.538 3.051-2.067 4.391-1.638 4.391-1.638 0.87 2.201 0.322 3.827 0.157 4.23 1.025 1.119 1.645 2.545 1.645 4.29 0 6.138-3.738 7.49-7.297 7.885 0.573 0.495 1.096 1.468 1.096 2.959v4.385c0 0.425 0.256 0.924 1.067 0.767 6.346-2.116 10.919-8.105 10.919-15.163 0-8.826-7.156-15.981-15.981-15.981z"></path>
						</svg>';
		}

		if ( 'google' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M16.275 13.962v4.899h8.104c-0.327 2.101-2.45 6.165-8.104 6.165-4.879 0-8.86-4.040-8.86-9.023 0-4.981 3.981-9.023 8.86-9.023 2.776 0 4.634 1.184 5.695 2.205l3.879-3.736c-2.491-2.327-5.716-3.736-9.574-3.736-7.9 0-14.29 6.39-14.29 14.29s6.39 14.29 14.29 14.29c8.247 0 13.72-5.798 13.72-13.963 0-0.939-0.104-1.654-0.227-2.368h-13.494z"></path>
						</svg>';
		}

		if ( 'hamburger' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M30.939 17.785h-29.876c-0.579 0-1.048-0.469-1.048-1.048v-1.482c0-0.579 0.469-1.048 1.048-1.048h29.876c0.579 0 1.048 0.469 1.048 1.048v1.482c0 0.579-0.469 1.048-1.048 1.048z"></path>
							<path d="M30.939 27.979h-29.876c-0.579 0-1.048-0.469-1.048-1.048v-1.482c0-0.579 0.469-1.048 1.048-1.048h29.876c0.579 0 1.048 0.469 1.048 1.048v1.482c0 0.579-0.469 1.048-1.048 1.048z"></path>
							<path d="M30.939 7.584h-29.876c-0.579 0-1.048-0.469-1.048-1.048v-1.482c0-0.579 0.469-1.048 1.048-1.048h29.876c0.579 0 1.048 0.469 1.048 1.048v1.482c0 0.579-0.469 1.048-1.048 1.048z"></path>
						</svg>';
		}

		if ( 'home' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M11.42 29.663c0 0.829-0.672 1.501-1.501 1.501-1.865 0-4.805 0-4.805 0-0.814 0-1.474-0.66-1.474-1.474v-13.912c0-0.531 0.286-1.021 0.748-1.283 2.293-1.299 9.187-5.205 11.108-6.294 0.311-0.177 0.693-0.177 1.004 0 1.922 1.089 8.815 4.994 11.108 6.294 0.462 0.262 0.747 0.752 0.747 1.283v13.912c0 0.814-0.66 1.474-1.474 1.474h-4.805c-0.829 0-1.501-0.672-1.501-1.501 0-2.012 0-5.562 0-7.581 0-0.838-0.679-1.517-1.517-1.517-1.696 0-4.426 0-6.122 0-0.838 0-1.517 0.679-1.517 1.517v7.581zM26.551 6.678l4.903 2.831c0.501 0.29 0.673 0.93 0.384 1.432l-0.741 1.284c-0.289 0.501-0.93 0.673-1.431 0.383 0 0-10.704-6.18-13.131-7.581-0.301-0.173-0.671-0.173-0.971 0-2.427 1.402-13.132 7.582-13.132 7.582-0.501 0.289-1.142 0.117-1.432-0.384l-0.741-1.284c-0.289-0.501-0.117-1.142 0.384-1.432l14.783-8.535c0.196-0.113 0.414-0.156 0.623-0.136 0.209-0.020 0.427 0.023 0.623 0.136l6.302 3.638v-2.352c0-0.579 0.469-1.048 1.048-1.048h1.483c0.579 0 1.048 0.469 1.048 1.048v4.418z"></path>
						</svg>';
		}

		if ( 'info' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M15.992 0.010c-8.826 0-15.992 7.166-15.992 15.992s7.166 15.993 15.992 15.993 15.992-7.166 15.992-15.993c0-8.826-7.166-15.992-15.992-15.992zM18.144 12.285v11.431c0 1.018 0.118 1.669 0.355 1.952s0.699 0.445 1.388 0.486v0.555h-7.79v-0.555c0.638-0.021 1.111-0.206 1.419-0.555 0.206-0.236 0.309-0.864 0.309-1.882v-8.423c0-1.018-0.118-1.669-0.355-1.952s-0.694-0.445-1.373-0.486v-0.571h6.047zM15.992 5.296c1.315 0 2.383 1.068 2.383 2.384s-1.068 2.383-2.383 2.383c-1.315 0-2.383-1.068-2.383-2.383s1.068-2.384 2.383-2.384z"></path>
						</svg>';
		}

		if ( 'instagram' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M24.544 5.541c-1.061 0-1.922 0.86-1.922 1.92s0.86 1.92 1.922 1.92c1.060 0 1.919-0.86 1.919-1.92s-0.859-1.92-1.919-1.92zM16.001 21.337c-2.946 0-5.334-2.387-5.334-5.334 0-2.946 2.388-5.334 5.334-5.334s5.334 2.388 5.334 5.334c0 2.947-2.388 5.334-5.334 5.334zM16.001 7.785c-4.538 0-8.217 3.679-8.217 8.217s3.679 8.219 8.217 8.219c4.538 0 8.217-3.679 8.217-8.219 0-4.538-3.679-8.217-8.217-8.217zM16.001 0c-4.346 0-4.89 0.018-6.597 0.096-5.812 0.267-9.042 3.491-9.308 9.308-0.079 1.708-0.097 2.252-0.097 6.598s0.019 4.891 0.096 6.598c0.267 5.812 3.491 9.041 9.308 9.308 1.708 0.077 2.252 0.096 6.598 0.096s4.892-0.019 6.598-0.096c5.806-0.267 9.044-3.491 9.307-9.308 0.079-1.707 0.097-2.252 0.097-6.598s-0.019-4.89-0.096-6.597c-0.261-5.806-3.49-9.041-9.307-9.308-1.708-0.079-2.254-0.097-6.6-0.097zM16.001 2.884c4.273 0 4.779 0.016 6.468 0.093 4.337 0.197 6.362 2.255 6.56 6.56 0.077 1.687 0.092 2.194 0.092 6.466 0 4.274-0.016 4.779-0.092 6.466-0.199 4.301-2.219 6.362-6.56 6.56-1.688 0.077-2.192 0.093-6.468 0.093-4.273 0-4.779-0.016-6.466-0.093-4.347-0.199-6.362-2.266-6.56-6.561-0.077-1.687-0.093-2.192-0.093-6.466 0-4.273 0.017-4.778 0.093-6.466 0.199-4.303 2.219-6.362 6.56-6.56 1.688-0.076 2.194-0.092 6.466-0.092z"></path>
						</svg>';
		}

		if ( 'linkedin' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M27.251 27.252h-4.738v-7.414c0-1.768-0.030-4.043-2.462-4.043-2.466 0-2.842 1.927-2.842 3.916v7.541h-4.733v-15.246h4.542v2.084h0.065c0.632-1.199 2.178-2.463 4.483-2.463 4.798 0 5.684 3.157 5.684 7.263v8.362zM7.133 9.923c-1.522 0-2.749-1.231-2.749-2.748s1.227-2.748 2.749-2.748c1.516 0 2.746 1.231 2.746 2.748s-1.231 2.748-2.746 2.748zM9.503 27.252h-4.743v-15.246h4.743v15.246zM29.611 0.029h-27.228c-1.3 0-2.356 1.031-2.356 2.304v27.339c0 1.272 1.057 2.305 2.356 2.305h27.228c1.303 0 2.363-1.033 2.363-2.305v-27.339c0-1.272-1.061-2.304-2.363-2.304z"></path>
						</svg>';
		}

		if ( 'messenger' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M15.999 0.018c-8.826 0-15.983 6.626-15.983 14.799 0 4.658 2.324 8.812 5.956 11.523v5.643l5.442-2.986c1.452 0.401 2.991 0.618 4.584 0.618 8.826 0 15.983-6.625 15.983-14.799 0-8.172-7.156-14.799-15.983-14.799zM17.588 19.947l-4.070-4.341-7.942 4.341 8.736-9.274 4.169 4.341 7.843-4.341-8.736 9.274z"></path>
						</svg>';
		}

		if ( 'patreon' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M20.499 0.669c-6.343 0-11.504 5.165-11.504 11.515 0 6.33 5.161 11.48 11.504 11.48 6.324 0 11.468-5.15 11.468-11.48 0-6.349-5.145-11.515-11.468-11.515z"></path>
							<path d="M0.018 0.669h5.617v30.672h-5.617v-30.672z"></path>
						</svg>';
		}

		if ( 'phone' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M28.791 31.121c-2.316 2.316-10.957 0.118-19.495-8.415-8.533-8.538-10.732-17.179-8.415-19.495 2.009-2.009 4.454-5.514 8.128-0.948s1.865 5.848-0.205 7.918c-1.445 1.45 1.578 5.115 4.741 8.277s6.826 6.186 8.272 4.735c2.076-2.070 3.352-3.874 7.923-0.205 4.571 3.675 1.061 6.124-0.948 8.133z"></path>
						</svg>';
		}

		if ( 'phone-o' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M19.761 19.398c-0.837-0.602-2.782-2.041-3.951-3.211-1.171-1.171-2.612-3.119-3.214-3.955 0.911-0.98 1.605-1.935 1.85-3.056 0.317-1.45 0.081-3.429-2.236-6.307-1.458-1.812-2.862-2.532-4.055-2.764-1.392-0.27-2.672 0.061-3.836 0.791-1.045 0.656-1.985 1.719-2.82 2.553-1.002 1.002-1.682 2.759-1.442 5.152 0.363 3.608 2.907 9.297 8.471 14.864 0 0 0.001 0.001 0.001 0.001 5.568 5.564 11.256 8.109 14.864 8.471 2.394 0.241 4.15-0.439 5.152-1.442v0c0.834-0.834 1.898-1.776 2.555-2.821 0.731-1.165 1.063-2.445 0.792-3.837-0.232-1.194-0.953-2.599-2.767-4.057-0-0-0.001-0.001-0.001-0.001-2.88-2.312-4.857-2.549-6.305-2.233-1.121 0.245-2.076 0.938-3.059 1.851zM26.682 28.632c-1.935 1.935-9.154 0.098-16.287-7.030-7.129-7.133-8.966-14.352-7.030-16.287 1.678-1.678 3.721-4.607 6.791-0.792s1.558 4.885-0.171 6.615c-1.207 1.212 1.319 4.273 3.96 6.915s5.703 5.168 6.91 3.956c1.734-1.73 2.8-3.237 6.619-0.171 3.819 3.070 0.886 5.116-0.792 6.795z"></path>
						</svg>';
		}

		if ( 'pinterest' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M15.989 0.042c-8.815 0-15.961 7.145-15.961 15.961 0 6.762 4.207 12.537 10.145 14.862-0.14-1.263-0.266-3.199 0.055-4.577 0.29-1.246 1.872-7.934 1.872-7.934s-0.477-0.956-0.477-2.37c0-2.219 1.287-3.876 2.888-3.876 1.361 0 2.019 1.022 2.019 2.248 0 1.369-0.872 3.416-1.322 5.314-0.377 1.589 0.796 2.885 2.363 2.885 2.837 0 5.016-2.991 5.016-7.309 0-3.821-2.745-6.493-6.666-6.493-4.541 0-7.207 3.406-7.207 6.926 0 1.372 0.529 2.843 1.188 3.642 0.131 0.159 0.149 0.298 0.111 0.459-0.121 0.504-0.391 1.588-0.444 1.809-0.070 0.292-0.231 0.355-0.534 0.214-1.994-0.929-3.24-3.842-3.24-6.184 0-5.035 3.658-9.659 10.546-9.659 5.537 0 9.84 3.946 9.84 9.218 0 5.501-3.468 9.928-8.282 9.928-1.618 0-3.138-0.839-3.658-1.833 0 0-0.801 3.049-0.994 3.794-0.36 1.387-1.333 3.125-1.984 4.185 1.494 0.462 3.080 0.712 4.726 0.712 8.814 0 15.961-7.147 15.961-15.961 0-8.816-7.147-15.961-15.961-15.961z"></path>
						</svg>';
		}

		if ( 'reddit' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M18.982 20.465c0.113 0.112 0.113 0.295 0 0.408-0.62 0.616-1.591 0.916-2.973 0.916l-0.011-0.003-0.011 0.003c-1.381 0-2.354-0.3-2.973-0.917-0.113-0.112-0.113-0.295 0-0.406 0.112-0.112 0.296-0.112 0.409 0 0.505 0.502 1.343 0.748 2.564 0.748l0.011 0.003 0.011-0.003c1.219 0 2.058-0.245 2.564-0.748 0.113-0.112 0.297-0.112 0.409 0zM14.398 17.243c0-0.676-0.552-1.225-1.229-1.225-0.678 0-1.23 0.549-1.23 1.225 0 0.674 0.552 1.223 1.23 1.223 0.677 0.001 1.229-0.548 1.229-1.223zM31.992 16.004c0 8.832-7.161 15.992-15.992 15.992s-15.992-7.161-15.992-15.992c0-8.832 7.161-15.992 15.992-15.992s15.992 7.161 15.992 15.992zM25.329 15.832c0-1.134-0.926-2.056-2.066-2.056-0.556 0-1.059 0.223-1.431 0.58-1.407-0.926-3.312-1.515-5.419-1.591l1.153-3.63 3.122 0.732-0.004 0.045c0 0.928 0.758 1.682 1.69 1.682s1.689-0.754 1.689-1.682c0-0.928-0.757-1.682-1.689-1.682-0.716 0-1.325 0.446-1.571 1.072l-3.365-0.789c-0.147-0.036-0.297 0.049-0.343 0.193l-1.286 4.049c-2.207 0.027-4.205 0.621-5.675 1.574-0.369-0.34-0.858-0.553-1.399-0.553-1.138 0.001-2.064 0.924-2.064 2.058 0 0.754 0.414 1.407 1.024 1.766-0.040 0.219-0.067 0.441-0.067 0.666 0 3.040 3.738 5.513 8.333 5.513s8.333-2.473 8.333-5.513c0-0.213-0.023-0.422-0.059-0.629 0.648-0.348 1.093-1.021 1.093-1.803zM18.836 16.020c-0.678 0-1.229 0.549-1.229 1.225 0 0.674 0.552 1.223 1.229 1.223s1.229-0.549 1.229-1.223c0-0.676-0.55-1.225-1.229-1.225z"></path>
						</svg>';
		}

		if ( 'rss' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M9.141 27.426c0 2.524-2.046 4.569-4.569 4.569s-4.569-2.046-4.569-4.569c0-2.524 2.046-4.569 4.569-4.569s4.569 2.046 4.569 4.569z"></path>
							<path d="M0.002 10.672v6.093c8.398 0 15.232 6.831 15.232 15.229h6.091c0-11.757-9.566-21.322-21.322-21.322z"></path>
							<path d="M0.002 0.010v6.093c14.279 0 25.894 11.613 25.894 25.892h6.091c0-17.637-14.347-31.985-31.985-31.985z"></path>
						</svg>';
		}

		if ( 'search' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M16 20.784c-1.094 0.463-2.259 0.698-3.462 0.698s-2.367-0.235-3.462-0.698c-1.059-0.448-2.011-1.090-2.829-1.908s-1.46-1.77-1.908-2.829c-0.463-1.094-0.698-2.259-0.698-3.462s0.235-2.367 0.698-3.462c0.448-1.059 1.090-2.011 1.908-2.829s1.77-1.46 2.829-1.908c1.094-0.463 2.259-0.698 3.462-0.698s2.367 0.235 3.462 0.698c1.059 0.448 2.011 1.090 2.829 1.908s1.46 1.77 1.908 2.829c0.463 1.094 0.698 2.259 0.698 3.462s-0.235 2.367-0.698 3.462c-0.448 1.059-1.090 2.011-1.908 2.829s-1.77 1.46-2.829 1.908zM31.661 29.088l-9.068-9.068c1.539-2.078 2.45-4.65 2.45-7.435 0-6.906-5.598-12.505-12.505-12.505s-12.505 5.598-12.505 12.505c0 6.906 5.598 12.505 12.505 12.505 2.831 0 5.442-0.941 7.537-2.526l9.055 9.055c0.409 0.409 1.073 0.409 1.482 0l1.048-1.048c0.409-0.409 0.409-1.073 0-1.482z"></path>
						</svg>';
		}

		if ( 'snapchat' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M7.773 6.051c-0.8 1.791-0.484 5-0.356 7.245-0.864 0.478-1.972-0.361-2.6-0.361-0.653 0-1.433 0.429-1.555 1.069-0.088 0.461 0.119 1.133 1.601 1.718 0.573 0.227 1.936 0.493 2.252 1.237 0.444 1.045-2.279 5.868-6.554 6.572-0.335 0.055-0.573 0.353-0.554 0.692 0.075 1.299 2.988 1.808 4.279 2.008 0.132 0.179 0.239 0.933 0.408 1.507 0.076 0.257 0.272 0.565 0.776 0.565 0.657 0 1.748-0.506 3.649-0.192 1.863 0.311 3.614 2.952 6.977 2.952 3.125 0 4.99-2.653 6.783-2.952 1.038-0.172 1.93-0.117 2.927 0.077 0.686 0.135 1.302 0.209 1.498-0.465 0.172-0.582 0.277-1.322 0.406-1.497 1.279-0.199 4.206-0.706 4.279-2.006 0.019-0.338-0.22-0.636-0.554-0.692-4.203-0.693-7.009-5.501-6.554-6.572 0.315-0.742 1.669-1.006 2.252-1.237 1.085-0.428 1.629-0.954 1.617-1.563-0.015-0.78-0.953-1.245-1.643-1.245-0.702 0-1.711 0.832-2.528 0.381 0.128-2.263 0.442-5.457-0.356-7.247-1.513-3.389-4.878-5.103-8.241-5.103-3.342 0-6.682 1.69-8.207 5.108z"></path>
						</svg>';
		}

		if ( 'soundcloud' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M4.055 15.408c-0.401-0.032-0.698-0.040-1.327 0.162v7.55c0.592 0.19 0.844 0.183 1.327 0.183v-7.895zM26.456 15.443c-0.271-3.765-3.191-6.741-6.791-6.741-1.356 0-2.607 0.435-3.666 1.164v13.435h12.063c2.133 0 3.863-1.849 3.863-4.122 0-2.964-2.877-5.006-5.469-3.736zM1.401 16.334c-0.804 0.726-1.327 1.797-1.327 3.010s0.523 2.284 1.327 3.010v-6.020zM5.382 23.302h1.327v-9.317c-0.414 0.608-0.737 1.289-0.918 2.037l-0.409-0.242v7.521zM10.691 23.302h1.327v-11.743c-0.674-0.105-0.827-0.066-1.327-0.013v11.756zM13.345 23.302h1.327v-12.242c-0.304 0.352-0.588 0.727-0.824 1.137l-0.503-0.244v11.349zM9.364 23.302h-1.327v-10.708c0.409-0.307 0.848-0.569 1.327-0.751v11.459z"></path>
						</svg>';
		}

		if ( 'spotify' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M25.386 14.203c-5.112-3.036-13.544-3.315-18.423-1.834-0.784 0.238-1.612-0.205-1.85-0.988s0.205-1.612 0.989-1.85c5.602-1.7 14.914-1.372 20.798 2.121 0.705 0.418 0.936 1.329 0.518 2.032-0.418 0.705-1.329 0.937-2.032 0.519zM25.219 18.699c-0.359 0.582-1.12 0.764-1.701 0.407-4.262-2.619-10.76-3.378-15.802-1.848-0.654 0.198-1.345-0.171-1.543-0.824-0.197-0.654 0.172-1.343 0.824-1.542 5.76-1.748 12.92-0.901 17.814 2.107 0.581 0.358 0.764 1.119 0.407 1.7zM23.278 23.017c-0.285 0.467-0.893 0.614-1.359 0.329-3.724-2.276-8.411-2.79-13.931-1.529-0.532 0.122-1.062-0.211-1.183-0.743s0.21-1.062 0.743-1.183c6.041-1.381 11.222-0.787 15.402 1.768 0.466 0.285 0.613 0.893 0.328 1.359zM16.006 0.144c-8.758 0-15.858 7.1-15.858 15.858s7.1 15.858 15.858 15.858c8.758 0 15.858-7.099 15.858-15.858s-7.1-15.858-15.858-15.858z"></path>
						</svg>';
		}

		if ( 'tiktok' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M29.662 12.949c-0.261 0.025-0.523 0.039-0.786 0.040-2.878 0-5.563-1.45-7.139-3.858v13.139c0 5.363-4.348 9.711-9.711 9.711s-9.711-4.348-9.711-9.711c0-5.363 4.348-9.711 9.711-9.711 0.203 0 0.401 0.018 0.6 0.031v4.785c-0.199-0.024-0.395-0.060-0.6-0.060-2.737 0-4.956 2.219-4.956 4.956s2.219 4.956 4.956 4.956c2.738 0 5.155-2.157 5.155-4.895l0.048-22.314h4.578c0.432 4.105 3.742 7.312 7.859 7.613v5.318z"></path>
						</svg>';
		}

		if ( 'times' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M29.094 5.43l-23.656 23.656c-0.41 0.41-1.074 0.41-1.483 0l-1.049-1.049c-0.41-0.41-0.41-1.073 0-1.483l23.656-23.656c0.41-0.41 1.073-0.41 1.483 0l1.049 1.049c0.41 0.409 0.41 1.073 0 1.483z"></path>
							<path d="M26.562 29.086l-23.656-23.656c-0.41-0.41-0.41-1.074 0-1.483l1.049-1.049c0.409-0.41 1.073-0.41 1.483 0l23.656 23.656c0.41 0.41 0.41 1.073 0 1.483l-1.049 1.049c-0.41 0.41-1.073 0.41-1.483 0z"></path>
						</svg>';
		}

		if ( 'tumblr' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M25.582 0.018h-19.197c-3.509 0-6.382 2.873-6.382 6.382v19.197c0 3.509 2.873 6.382 6.382 6.382h19.197c3.509 0 6.382-2.873 6.382-6.382v-19.197c0-3.509-2.873-6.382-6.382-6.382zM21.489 25.953h-3.369c-3.033 0-5.293-1.561-5.293-5.293v-5.978h-2.756v-3.237c3.033-0.788 4.301-3.398 4.447-5.658h3.15v5.133h3.674v3.762h-3.674v5.205c0 1.56 0.787 2.1 2.041 2.1h1.779v3.966z"></path>
						</svg>';
		}

		if ( 'twitter' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M18.979,13.55l11.656,-13.55l-2.762,0l-10.121,11.765l-8.084,-11.765l-9.324,0l12.225,17.791l-12.225,14.209l2.762,0l10.689,-12.424l8.537,12.424l9.324,0l-12.678,-18.45l0.001,-0Zm-3.784,4.398l-1.238,-1.772l-9.855,-14.097l4.243,0l7.953,11.377l1.238,1.771l10.339,14.788l-4.243,0l-8.437,-12.067l0,-0Z"></path>
						</svg>';
		}

		if ( 'user' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M11.524 15.093c-2.265-1.466-3.764-4.014-3.764-6.909 0-4.539 3.685-8.224 8.224-8.224s8.224 3.685 8.224 8.224c0 2.896-1.499 5.444-3.764 6.909 3.9 2.446 6.762 8.003 7.278 14.66 0.045 0.558-0.146 1.11-0.526 1.522s-0.914 0.645-1.474 0.645c-7.429 0.001-12.758 0.001-19.476 0.001-0.56 0-1.095-0.234-1.475-0.646s-0.571-0.964-0.526-1.522c0.517-6.657 3.379-12.214 7.279-14.66z"></path>
						</svg>';
		}

		if ( 'user-o' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M9.444 15.186c-3.030 2.804-5.228 7.599-5.665 13.223-0.075 0.945 0.248 1.879 0.891 2.576s1.55 1.094 2.499 1.094c6.093 0 10.927 0 17.666-0.001 0.949 0 1.854-0.397 2.498-1.094s0.966-1.63 0.891-2.575c-0.436-5.624-2.634-10.419-5.664-13.223 1.538-1.621 2.481-3.811 2.481-6.219 0-4.989-4.051-9.040-9.040-9.040s-9.040 4.051-9.040 9.040c0 2.409 0.944 4.598 2.481 6.219zM12.796 16.572c0.457-0.287 0.736-0.787 0.74-1.327s-0.268-1.044-0.721-1.337c-1.62-1.048-2.693-2.87-2.693-4.941 0-3.246 2.635-5.881 5.881-5.881s5.881 2.635 5.881 5.881c0 2.071-1.074 3.893-2.693 4.941-0.453 0.293-0.725 0.798-0.721 1.337s0.283 1.040 0.74 1.327c3.2 2.007 5.442 6.619 5.866 12.082 0 0.001 0 0.003 0 0.004 0.005 0.067-0.017 0.133-0.063 0.182s-0.11 0.077-0.177 0.077c-0 0-0 0-0 0-6.739 0.001-11.573 0.001-17.666 0.001-0.067 0-0.132-0.028-0.178-0.078s-0.069-0.116-0.063-0.183c0-0.001 0-0.002 0-0.004 0.424-5.463 2.667-10.075 5.867-12.082z"></path>
						</svg>';
		}

		if ( 'vimeo' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M26.904 10.635c-0.103 2.249-1.673 5.327-4.712 9.236-3.141 4.081-5.799 6.123-7.972 6.123-1.347 0-2.487-1.242-3.418-3.729-0.622-2.279-1.243-4.558-1.864-6.837-0.692-2.485-1.433-3.729-2.226-3.729-0.173 0-0.778 0.364-1.813 1.088l-1.087-1.4c1.14-1.002 2.265-2.003 3.371-3.006 1.521-1.313 2.663-2.005 3.424-2.074 1.797-0.173 2.904 1.056 3.319 3.686 0.449 2.838 0.76 4.603 0.934 5.294 0.519 2.355 1.089 3.531 1.712 3.531 0.483 0 1.21-0.764 2.179-2.292 0.967-1.527 1.485-2.689 1.555-3.488 0.138-1.318-0.38-1.979-1.555-1.979-0.553 0-1.124 0.127-1.71 0.379 1.136-3.718 3.305-5.524 6.507-5.421 2.374 0.070 3.494 1.609 3.356 4.618zM28.776 0.018h-25.576c-1.766 0-3.197 1.431-3.197 3.197v25.576c0 1.766 1.432 3.197 3.197 3.197h25.576c1.765 0 3.197-1.431 3.197-3.197v-25.576c0-1.766-1.431-3.197-3.197-3.197z"></path>
						</svg>';
		}

		if ( 'whatsapp' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M0.076 31.995l2.248-8.213c-1.387-2.404-2.116-5.13-2.115-7.924 0.004-8.737 7.114-15.847 15.85-15.847 4.239 0.001 8.219 1.653 11.212 4.648 2.992 2.996 4.639 6.978 4.638 11.213-0.004 8.738-7.114 15.848-15.85 15.848-2.652-0.001-5.265-0.666-7.58-1.93l-8.403 2.204zM8.868 26.921c2.234 1.326 4.366 2.12 7.186 2.122 7.261 0 13.175-5.909 13.179-13.174 0.003-7.279-5.884-13.18-13.168-13.183-7.266 0-13.176 5.909-13.179 13.172-0.001 2.965 0.868 5.186 2.327 7.508l-1.331 4.862 4.987-1.307zM24.044 19.639c-0.099-0.165-0.363-0.264-0.76-0.462-0.396-0.199-2.343-1.157-2.707-1.289-0.363-0.132-0.626-0.199-0.892 0.199-0.264 0.396-1.023 1.289-1.254 1.553s-0.462 0.297-0.858 0.099c-0.396-0.199-1.673-0.616-3.185-1.966-1.177-1.050-1.972-2.347-2.203-2.744-0.231-0.396-0.024-0.61 0.173-0.808 0.179-0.177 0.396-0.462 0.594-0.694 0.201-0.229 0.267-0.394 0.4-0.66 0.132-0.264 0.067-0.496-0.033-0.694-0.1-0.197-0.892-2.147-1.221-2.94-0.322-0.772-0.649-0.668-0.892-0.68l-0.76-0.013c-0.264 0-0.693 0.099-1.056 0.496s-1.386 1.354-1.386 3.304c0 1.95 1.419 3.833 1.617 4.097 0.199 0.264 2.792 4.265 6.765 5.98 0.945 0.408 1.683 0.652 2.258 0.834 0.949 0.301 1.812 0.259 2.495 0.157 0.761-0.113 2.343-0.958 2.673-1.883 0.331-0.926 0.331-1.719 0.231-1.884z"></path>
						</svg>';
		}

		if ( 'xing' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M19.848 31.992l-7.095-12.883 10.755-19.102h6.574l-10.754 19.102 7.023 12.883h-6.504zM10.131 6.67h-6.582l3.689 6.322-5.484 9.67h6.549l5.487-9.69-3.658-6.302z"></path>
						</svg>';
		}

		if ( 'yelp' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M7.102 15.043l6.172 3.010c1.187 0.579 1.004 2.325-0.278 2.644l-6.662 1.661c-0.831 0.207-1.652-0.366-1.75-1.217-0.21-1.82-0.003-3.617 0.557-5.281 0.272-0.808 1.195-1.191 1.961-0.817z"></path>
							<path d="M9.573 27.748l4.594-5.102c0.884-0.982 2.511-0.324 2.465 0.996l-0.24 6.864c-0.030 0.855-0.807 1.489-1.651 1.347-1.775-0.298-3.454-0.984-4.919-1.991-0.704-0.484-0.821-1.478-0.249-2.113z"></path>
							<path d="M20.464 20.958l6.529 2.122c0.813 0.264 1.209 1.184 0.841 1.955-0.766 1.605-1.877 3.038-3.265 4.184-0.66 0.545-1.649 0.385-2.103-0.341l-3.64-5.825c-0.7-1.12 0.381-2.503 1.637-2.095z"></path>
							<path d="M27.104 16.793l-6.6 1.892c-1.27 0.364-2.301-1.056-1.563-2.151l3.84-5.692c0.477-0.707 1.468-0.837 2.107-0.273 1.316 1.161 2.394 2.614 3.123 4.295 0.34 0.786-0.084 1.693-0.907 1.929z"></path>
							<path d="M11.682 0.777c-1.188 0.318-2.321 0.739-3.393 1.25-0.744 0.355-1.025 1.27-0.613 1.984l6.451 11.173c0.724 1.254 2.64 0.74 2.64-0.707v-12.902c0-0.824-0.701-1.476-1.522-1.412-1.184 0.093-2.375 0.296-3.563 0.614z"></path>
						</svg>';
		}

		if ( 'youtube' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M12.803 20.8v-9.595l8.309 4.798-8.309 4.797zM31.324 8.305c-0.368-1.376-1.452-2.46-2.828-2.828-2.494-0.668-12.495-0.668-12.495-0.668s-10.001 0-12.495 0.668c-1.376 0.368-2.46 1.452-2.828 2.828-0.668 2.494-0.668 7.698-0.668 7.698s0 5.204 0.668 7.698c0.368 1.376 1.452 2.46 2.828 2.828 2.494 0.668 12.495 0.668 12.495 0.668s10.001 0 12.495-0.668c1.376-0.368 2.46-1.452 2.828-2.828 0.668-2.494 0.668-7.698 0.668-7.698s0-5.204-0.668-7.698z"></path>
						</svg>';
		}

		if ( 'bag-2' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M30.467,1.881c0,0 0.454,14.306 0.718,22.625c0.062,1.953 -0.671,3.847 -2.029,5.249c-1.359,1.402 -3.229,2.196 -5.182,2.196l-16.161,-0c-1.953,-0 -3.823,-0.794 -5.181,-2.196c-1.359,-1.402 -2.092,-3.296 -2.03,-5.249c0.264,-8.319 0.718,-22.625 0.718,-22.625c0.027,-0.851 0.725,-1.528 1.577,-1.528l25.993,0c0.852,0 1.55,0.677 1.577,1.528l0,0Zm-5.804,1.629c0.244,0.278 0.393,0.642 0.393,1.042c0,5.544 -4.152,9.969 -9.162,9.969c-5.011,0 -9.163,-4.425 -9.163,-9.969c0,-0.4 0.149,-0.764 0.394,-1.042l-2.699,-0l-0.669,21.098c-0.035,1.098 0.377,2.162 1.142,2.951c0.764,0.789 1.816,1.235 2.914,1.235l16.161,0c1.099,0 2.15,-0.446 2.915,-1.235c0.764,-0.789 1.176,-1.853 1.141,-2.951c-0.212,-6.671 -0.545,-17.191 -0.669,-21.098l-2.698,-0Zm-2.37,-0l-12.799,-0c0.245,0.278 0.393,0.642 0.393,1.042c0,3.72 2.645,6.813 6.007,6.813c3.361,-0 6.006,-3.093 6.006,-6.813c-0,-0.4 0.148,-0.764 0.393,-1.042l0,-0Z"></path>
						</svg>';
		}

		if ( 'calendar' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M9.06,2.345l-3.617,0c-2.298,0 -4.164,1.866 -4.164,4.163l0,21.324c0,2.297 1.866,4.164 4.164,4.164l21.119,-0c2.298,-0 4.164,-1.867 4.164,-4.164l-0,-21.324c-0,-2.297 -1.866,-4.163 -4.164,-4.163l-3.618,0l0,-1.37c0,-0.533 -0.4,-0.965 -0.893,-0.965l-1.265,0c-0.493,0 -0.893,0.432 -0.893,0.965l-0,1.37l-7.781,0l0,-1.37c0,-0.533 -0.4,-0.965 -0.894,-0.965l-1.264,0c-0.493,0 -0.894,0.432 -0.894,0.965l0,1.37l0,0Zm18.618,11.071l0,14.416c0,0.616 -0.5,1.115 -1.116,1.115l-21.119,-0c-0.616,-0 -1.116,-0.499 -1.116,-1.115l-0,-14.416l23.351,0l0,0Zm-16.35,9.009l0,3.791c0,0.201 -0.161,0.361 -0.361,0.361l-3.791,0c-0.199,0 -0.361,-0.16 -0.361,-0.361l0,-3.791c0,-0.199 0.162,-0.361 0.361,-0.361l3.791,-0c0.2,-0 0.361,0.162 0.361,0.361Zm6.931,-0l-0,3.791c-0,0.201 -0.162,0.361 -0.361,0.361l-3.791,0c-0.2,0 -0.361,-0.16 -0.361,-0.361l-0,-3.791c-0,-0.199 0.161,-0.361 0.361,-0.361l3.791,-0c0.199,-0 0.361,0.162 0.361,0.361Zm-6.931,-6.606l0,3.792c0,0.198 -0.161,0.361 -0.361,0.361l-3.791,-0c-0.199,-0 -0.361,-0.163 -0.361,-0.361l0,-3.792c0,-0.2 0.162,-0.361 0.361,-0.361l3.791,0c0.2,0 0.361,0.161 0.361,0.361Zm6.931,0l-0,3.792c-0,0.198 -0.162,0.361 -0.361,0.361l-3.791,-0c-0.2,-0 -0.361,-0.163 -0.361,-0.361l-0,-3.792c-0,-0.2 0.161,-0.361 0.361,-0.361l3.791,0c0.199,0 0.361,0.161 0.361,0.361Zm6.931,0l-0,3.792c-0,0.198 -0.162,0.361 -0.361,0.361l-3.792,-0c-0.199,-0 -0.361,-0.163 -0.361,-0.361l0,-3.792c0,-0.2 0.162,-0.361 0.361,-0.361l3.792,0c0.199,0 0.361,0.161 0.361,0.361Zm-2.246,-10.426l3.618,0c0.616,0 1.116,0.499 1.116,1.115l0,3.848l-23.351,0l-0,-3.848c-0,-0.616 0.5,-1.115 1.116,-1.115l3.617,0l0,1.097c0,0.533 0.401,0.964 0.894,0.964l1.264,-0c0.494,-0 0.894,-0.431 0.894,-0.964l0,-1.097l7.781,0l-0,1.097c-0,0.533 0.4,0.964 0.893,0.964l1.265,-0c0.493,-0 0.893,-0.431 0.893,-0.964l0,-1.097l0,0Z"></path>
						</svg>';
		}

		if ( 'loader' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M27.74,25.619c0.177,0.102 0.298,0.278 0.33,0.48c0.032,0.201 -0.028,0.406 -0.164,0.558c-2.926,3.266 -7.177,5.32 -11.907,5.32c-8.828,0 -15.985,-7.156 -15.985,-15.985c-0,-8.828 7.157,-15.985 15.985,-15.985c4.73,0 8.981,2.055 11.907,5.321c0.136,0.152 0.196,0.357 0.164,0.558c-0.032,0.202 -0.153,0.378 -0.33,0.48c-0.619,0.357 -1.503,0.868 -2.039,1.177c-0.27,0.156 -0.612,0.107 -0.828,-0.119c-2.268,-2.348 -5.431,-3.828 -8.874,-3.828c-6.734,0 -12.396,5.662 -12.396,12.396c-0,6.735 5.662,12.397 12.396,12.397c3.443,-0 6.606,-1.48 8.875,-3.828c0.215,-0.225 0.557,-0.274 0.827,-0.118c0.536,0.308 1.42,0.819 2.039,1.176Z"></path><circle cx="31.955" cy="15.992" r="0.03" style="fill:none;"/>
						</svg>';
		}

		if ( 'document' === $icon ) {
			$output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" role="img" aria-hidden="true">
							<path d="M9.06,2.345l-3.617,0c-2.298,0 -4.164,1.866 -4.164,4.163l0,21.324c0,2.297 1.866,4.164 4.164,4.164l21.119,-0c2.298,-0 4.164,-1.867 4.164,-4.164l-0,-21.324c-0,-2.297 -1.866,-4.163 -4.164,-4.163l-3.618,0l0,-1.37c0,-0.533 -0.4,-0.965 -0.893,-0.965l-1.265,0c-0.493,0 -0.893,0.432 -0.893,0.965l-0,1.37l-7.781,0l0,-1.37c0,-0.533 -0.4,-0.965 -0.894,-0.965l-1.264,0c-0.493,0 -0.894,0.432 -0.894,0.965l0,1.37l0,0Zm18.618,11.071l0,14.416c0,0.616 -0.5,1.115 -1.116,1.115l-21.119,-0c-0.616,-0 -1.116,-0.499 -1.116,-1.115l-0,-14.416l23.351,0l0,0Zm-16.35,9.009l0,3.791c0,0.201 -0.161,0.361 -0.361,0.361l-3.791,0c-0.199,0 -0.361,-0.16 -0.361,-0.361l0,-3.791c0,-0.199 0.162,-0.361 0.361,-0.361l3.791,-0c0.2,-0 0.361,0.162 0.361,0.361Zm6.931,-0l-0,3.791c-0,0.201 -0.162,0.361 -0.361,0.361l-3.791,0c-0.2,0 -0.361,-0.16 -0.361,-0.361l-0,-3.791c-0,-0.199 0.161,-0.361 0.361,-0.361l3.791,-0c0.199,-0 0.361,0.162 0.361,0.361Zm-6.931,-6.606l0,3.792c0,0.198 -0.161,0.361 -0.361,0.361l-3.791,-0c-0.199,-0 -0.361,-0.163 -0.361,-0.361l0,-3.792c0,-0.2 0.162,-0.361 0.361,-0.361l3.791,0c0.2,0 0.361,0.161 0.361,0.361Zm6.931,0l-0,3.792c-0,0.198 -0.162,0.361 -0.361,0.361l-3.791,-0c-0.2,-0 -0.361,-0.163 -0.361,-0.361l-0,-3.792c-0,-0.2 0.161,-0.361 0.361,-0.361l3.791,0c0.199,0 0.361,0.161 0.361,0.361Zm6.931,0l-0,3.792c-0,0.198 -0.162,0.361 -0.361,0.361l-3.792,-0c-0.199,-0 -0.361,-0.163 -0.361,-0.361l0,-3.792c0,-0.2 0.162,-0.361 0.361,-0.361l3.792,0c0.199,0 0.361,0.161 0.361,0.361Zm-2.246,-10.426l3.618,0c0.616,0 1.116,0.499 1.116,1.115l0,3.848l-23.351,0l-0,-3.848c-0,-0.616 0.5,-1.115 1.116,-1.115l3.617,0l0,1.097c0,0.533 0.401,0.964 0.894,0.964l1.264,-0c0.494,-0 0.894,-0.431 0.894,-0.964l0,-1.097l7.781,0l-0,1.097c-0,0.533 0.4,0.964 0.893,0.964l1.265,-0c0.493,-0 0.893,-0.431 0.893,-0.964l0,-1.097l0,0Z"></path>
						</svg>';
		}

		$classes = array(
			'wpbf-icon',
			'wpbf-icon-' . $icon,
		);

		$output = sprintf(
			'<span class="%1$s">%2$s</span>',
			implode( ' ', $classes ),
			$output
		);

		return $output;

	}

}

/**
 * Add a class to our main navigation.
 */
function wpbf_navigation_classes() {
	echo apply_filters( 'wpbf_navigation_classes', 'wpbf-navigation' );
}

/**
 * Inner content open.
 *
 * @param boolean $echo Determine wether result should return or echo.
 */
function wpbf_inner_content( $echo = true ) {

	if ( is_singular() ) {

		$options = get_post_meta( get_the_ID(), 'wpbf_options', true );
		$options = is_array( $options ) ? $options : array();

		// Check if template is set to full width.
		$fullwidth = in_array( 'full-width', $options, true );

		// Check if template is set to contained.
		$contained = in_array( 'contained', $options, true );

		// Check if template is set to custom width.
		$custom_width       = in_array( 'custom-width', $options, true );
		$custom_width_value = isset( $options['custom_width_value'] ) ? $options['custom_width_value'] : '';
		$custom_width       = $custom_width && $custom_width_value ? ' style="max-width: ' . $custom_width_value . '"' : '';

		// Construct inner content wrapper.
		$inner_content = $fullwidth ? '' : apply_filters( 'wpbf_inner_content', '<div id="inner-content" class="wpbf-container wpbf-container-center wpbf-padding-medium"' . $custom_width . '>' );

		// Check if Premium Add-On is active and template is not set to contained.
		if ( wpbf_is_premium() && ! $contained && ! $custom_width ) {

			$wpbf_settings = get_option( 'wpbf_settings' );

			// Get array of post types that are set to full width under Appearance > Theme Settings > Global Templat Settings.
			$fullwidth_global = isset( $wpbf_settings['wpbf_fullwidth_global'] ) ? $wpbf_settings['wpbf_fullwidth_global'] : array();

			// If current post type has been set to full-width globally, set $inner_content to empty.
			$inner_content = $fullwidth_global && in_array( get_post_type( get_the_ID() ), $fullwidth_global ) ? '' : $inner_content;

		}
	} else {

		// On archives, we only add the wpbf_inner_content filter.
		$inner_content = apply_filters( 'wpbf_inner_content', '<div id="inner-content" class="wpbf-container wpbf-container-center wpbf-padding-medium">' );

	}

	if ( $echo ) {
		echo $inner_content;
	} else {
		return $inner_content;
	}

}

/**
 * Inner content close.
 */
function wpbf_inner_content_close() {

	if ( is_singular() ) {

		$options = get_post_meta( get_the_ID(), 'wpbf_options', true );
		$options = is_array( $options ) ? $options : array();

		$fullwidth = in_array( 'full-width', $options, true );

		$contained = in_array( 'contained', $options, true );

		$custom_width = in_array( 'custom-width', $options, true );

		$inner_content_close = $fullwidth ? '' : '</div>';

		if ( wpbf_is_premium() && ! $contained && ! $custom_width ) {

			$wpbf_settings = get_option( 'wpbf_settings' );

			$fullwidth_global = isset( $wpbf_settings['wpbf_fullwidth_global'] ) ? $wpbf_settings['wpbf_fullwidth_global'] : array();

			$inner_content_close = $fullwidth_global && in_array( get_post_type( get_the_ID() ), $fullwidth_global ) ? '' : $inner_content_close;

		}
	} else {

		$inner_content_close = '</div>';

	}

	echo $inner_content_close;

}

/**
 * Title.
 */
function wpbf_title() {

	$options = get_post_meta( get_the_ID(), 'wpbf_options', true );
	$options = is_array( $options ) ? $options : array();

	$removetitle = in_array( 'remove-title', $options, true );

	$title = $removetitle ? false : '<h1 class="entry-title" itemprop="headline">' . get_the_title() . '</h1>';

	if ( wpbf_is_premium() ) {

		$wpbf_settings = get_option( 'wpbf_settings' );

		$removetitle_global = isset( $wpbf_settings['wpbf_removetitle_global'] ) ? $wpbf_settings['wpbf_removetitle_global'] : array();

		$title = $removetitle_global && in_array( get_post_type( get_the_ID() ), $removetitle_global ) ? false : $title;

	}

	// Use this filter if you want to hide the title from specific pages, etc.
	// To actually change the title itself please filter the_title() directly.
	$title = apply_filters( 'wpbf_title', $title );

	if ( $title ) {

		do_action( 'wpbf_before_page_title' );

		echo $title;

		do_action( 'wpbf_after_page_title' );

	}

}

/**
 * Disable header.
 */
function wpbf_remove_header() {

	// Stop here if we're on archives.
	if ( ! is_singular() ) {
		return;
	}

	$options = get_post_meta( get_the_ID(), 'wpbf_options', true );
	$options = is_array( $options ) ? $options : array();

	// Check if header is disabled.
	$remove_header = in_array( 'remove-header', $options, true );

	// Remove header if disabled.
	if ( $remove_header ) {
		remove_action( 'wpbf_header', 'wpbf_do_header' );
	}

}
add_action( 'wp', 'wpbf_remove_header' );

/**
 * Disable footer.
 */
function wpbf_remove_footer() {

	// Stop here if we're on archives.
	if ( ! is_singular() ) {
		return;
	}

	$options = get_post_meta( get_the_ID(), 'wpbf_options', true );
	$options = is_array( $options ) ? $options : array();

	// Check if footer is disabled.
	$remove_footer = in_array( 'remove-footer', $options, true );

	// Remove footer if disabled.
	if ( $remove_footer ) {
		remove_action( 'wpbf_footer', 'wpbf_do_footer' );
		remove_action( 'wpbf_before_footer', 'wpbf_custom_footer' );
	}

}
add_action( 'wp', 'wpbf_remove_footer' );

/**
 * ScrollTop.
 */
function wpbf_scrolltop() {

	if ( get_theme_mod( 'layout_scrolltop' ) ) {

		$scrolltop = get_theme_mod( 'scrolltop_value', 400 );

		if ( wpbf_svg_enabled() ) {
			$icon = wpbf_svg( 'arrow-up' );
		} else {
			$icon = '<i class="wpbff wpbff-arrow-up"></i>';
		}

		echo '<a class="scrolltop" rel="nofollow" href="javascript:void(0)" data-scrolltop-value="' . (int) $scrolltop . '">';
		echo '<span class="screen-reader-text">' . __( 'Scroll to Top', 'page-builder-framework' ) . '</span>';
		echo $icon;
		echo '</a>';

	}

}
add_action( 'wp_footer', 'wpbf_scrolltop' );

/**
 * Archive Class
 *
 * Add unique class to any existing archive type.
 *
 * wpbf-archive-content
 * + wpbf-{post-type}-archive
 * + wpbf-{archive-type}-content (for post archives)
 *
 * + wpbf-{post-type}-archive-content (for cpt archives)
 * + wpbf-{post-type}-taxonomy-content (for cpt-related taxonomies)
 *
 * @return string The archive class.
 */
function wpbf_archive_class() {

	$archive_class = '';

	if ( is_date() ) {
		$archive_class = ' wpbf-archive-content wpbf-post-archive wpbf-date-content';
	} elseif ( is_category() ) {
		$archive_class = ' wpbf-archive-content wpbf-post-archive wpbf-category-content';
	} elseif ( is_tag() ) {
		$archive_class = ' wpbf-archive-content wpbf-post-archive wpbf-tag-content';
	} elseif ( is_author() ) {
		$archive_class = ' wpbf-archive-content wpbf-post-archive wpbf-author-content';
	} elseif ( is_home() ) {
		$archive_class = ' wpbf-archive-content wpbf-post-archive wpbf-blog-content';
	} elseif ( is_search() ) {
		$archive_class = ' wpbf-archive-content wpbf-post-archive wpbf-search-content';
	} elseif ( is_post_type_archive() ) {

		$post_type = get_post_type();

		// Stop here if no post has been found.
		if ( ! $post_type ) {
			return $archive_class;
		}

		$archive_class  = ' wpbf-archive-content';
		$archive_class .= ' wpbf-' . $post_type . '-archive';
		$archive_class .= ' wpbf-' . $post_type . '-archive-content';

	} elseif ( is_tax() ) {

		$post_type = get_post_type();

		// Stop here if no post has been found.
		if ( ! $post_type ) {
			return $archive_class;
		}

		$archive_class  = ' wpbf-archive-content';
		$archive_class .= ' wpbf-' . $post_type . '-archive';
		$archive_class .= ' wpbf-' . $post_type . '-archive-content';
		$archive_class .= ' wpbf-' . $post_type . '-taxonomy-content';

	}

	return apply_filters( 'wpbf_archive_class', $archive_class );

}

/**
 * Singular class.
 *
 * @return string The singular class.
 */
function wpbf_singular_class() {

	if ( is_singular( 'post' ) ) {
		$singular_class = ' wpbf-single-content wpbf-post-content';
	} elseif ( is_attachment() ) {
		$singular_class = ' wpbf-single-content wpbf-attachment-content';
	} elseif ( is_page() ) {
		$singular_class = ' wpbf-single-content wpbf-page-content';
	} elseif ( is_404() ) {
		$singular_class = ' wpbf-single-content wpbf-404-content';
	} else {
		$post_type      = get_post_type();
		$singular_class = ' wpbf-single-content wpbf-' . $post_type . '-content';
	}

	return apply_filters( 'wpbf_singular_class', $singular_class );

}

/**
 * Archive header.
 */
function wpbf_archive_header() {

	if ( is_author() ) {

		?>
		<section class="wpbf-author-box">
			<h1 class="page-title"><span class="vcard"><?php echo get_the_author(); ?></span></h1>
			<p><?php echo wp_kses_post( get_the_author_meta( 'description' ) ); ?></p>
			<?php echo get_avatar( get_the_author_meta( 'email' ), 120 ); ?>
		</section>
		<?php

	} elseif ( is_home() ) {

		$blog_title = apply_filters( 'wpbf_blog_page_title', '' );

		if ( ! empty( $blog_title ) ) {

			do_action( 'wpbf_before_page_title' );

			echo '<h1 class="page-title">';

			echo $blog_title;

			echo '</h1>';

			do_action( 'wpbf_after_page_title' );

		}
	} elseif ( is_search() ) {

		do_action( 'wpbf_before_page_title' );

		echo '<h1 class="page-title">';

		echo apply_filters(
			'wpbf_search_page_title',
			sprintf(
				/* translators: Search query */
				__( 'Search Results for: %s', 'page-builder-framework' ),
				'<span>' . get_search_query() . '</span>'
			)
		);

		echo '</h1>';

		do_action( 'wpbf_after_page_title' );

	} else {

		if ( get_the_archive_title() ) {

			do_action( 'wpbf_before_page_title' );

			the_archive_title( '<h1 class="page-title">', '</h1>' );

			do_action( 'wpbf_after_page_title' );

		}

		the_archive_description( '<div class="taxonomy-description">', '</div>' );

	}

}

/**
 * Archive headline.
 *
 * @param string $title The archive headline.
 *
 * @return string The updated archive headline.
 */
function wpbf_archive_title( $title ) {

	$archive_headline = get_theme_mod( 'archive_headline' );

	if ( is_category() ) {

		if ( 'hide_prefix' === $archive_headline ) {
			$title = single_cat_title( '', false );
		} elseif ( 'hide' === $archive_headline ) {
			$title = false;
		}
	} elseif ( is_tag() ) {

		if ( 'hide_prefix' === $archive_headline ) {
			$title = single_tag_title( '', false );
		} elseif ( 'hide' === $archive_headline ) {
			$title = false;
		}
	} elseif ( is_date() ) {

		$date = get_the_date( 'F Y' );
		if ( is_year() ) {
			$date = get_the_date( 'Y' );
		}

		if ( is_day() ) {
			$date = get_the_date( 'F j, Y' );
		}

		if ( 'hide_prefix' === $archive_headline ) {
			$title = $date;
		} elseif ( 'hide' === $archive_headline ) {
			$title = false;
		}
	} elseif ( is_post_type_archive() ) {

		if ( 'hide_prefix' === $archive_headline ) {
			$title = post_type_archive_title( '', false );
		} elseif ( 'hide' === $archive_headline ) {
			$title = false;
		}
	} elseif ( is_tax() ) {

		if ( 'hide_prefix' === $archive_headline ) {
			$title = single_term_title( '', false );
		} elseif ( 'hide' === $archive_headline ) {
			$title = false;
		}
	}

	return $title;

}
add_filter( 'get_the_archive_title', 'wpbf_archive_title', 10 );

/**
 * Post links.
 *
 * Display the post navigation on posts.
 */
function wpbf_do_post_links() {

	// Filter to allow conditional display.
	if ( ! apply_filters( 'wpbf_display_post_links', true ) ) {
		return;
	}

	do_action( 'wpbf_before_post_links' );

	?>

	<nav class="post-links wpbf-clearfix" aria-label="<?php _e( 'Post Navigation', 'page-builder-framework' ); ?>">

		<span class="screen-reader-text"><?php _e( 'Post Navigation', 'page-builder-framework' ); ?></span>

		<?php
		previous_post_link( '<span class="previous-post-link">%link</span>', apply_filters( 'wpbf_previous_post_link', __( '&larr; Previous Post', 'page-builder-framework' ) ) );
		next_post_link( '<span class="next-post-link">%link</span>', apply_filters( 'wpbf_next_post_link', __( 'Next Post &rarr;', 'page-builder-framework' ) ) );
		?>

	</nav>

	<?php

	do_action( 'wpbf_after_post_links' );

}
add_action( 'wpbf_post_links', 'wpbf_do_post_links' );

/**
 * Posts pagination.
 *
 * Display the posts pagination on archives.
 */
function wpbf_do_posts_pagination() {

	the_posts_pagination(
		array(
			'mid_size'  => apply_filters( 'wpbf_posts_pagination_size', 2 ),
			'prev_text' => apply_filters( 'wpbf_posts_navigation_prev_text', __( '&larr; Previous', 'page-builder-framework' ) ),
			'next_text' => apply_filters( 'wpbf_posts_navigation_next_text', __( 'Next &rarr;', 'page-builder-framework' ) ),
		)
	);

}
add_action( 'wpbf_posts_pagination', 'wpbf_do_posts_pagination' );

if ( ! function_exists( 'wpbf_has_responsive_breakpoints' ) ) {

	/**
	 * Responsive breakpoints.
	 *
	 * Simple check if responsive breakpoints are set.
	 *
	 * @return boolean.
	 */
	function wpbf_has_responsive_breakpoints() {

		// There can't be responsive breakpoints if there's no Premium Add-On.
		if ( ! wpbf_is_premium() ) {
			return false;
		}

		$wpbf_settings = get_option( 'wpbf_settings' );

		// Check if custom breakpoints are set, otherwise return false.
		if ( ! empty( $wpbf_settings['wpbf_breakpoint_medium'] ) || ! empty( $wpbf_settings['wpbf_breakpoint_desktop'] ) || ! empty( $wpbf_settings['wpbf_breakpoint_mobile'] ) ) {
			return true;
		} else {
			return false;
		}

	}
}

/**
 * Render right sidebar.
 */
function wpbf_do_sidebar_right() {

	if ( 'right' === wpbf_sidebar_layout() ) {
		get_sidebar();
	}

}
add_action( 'wpbf_sidebar_right', 'wpbf_do_sidebar_right' );

/**
 * Render left sidebar.
 */
function wpbf_do_sidebar_left() {

	if ( 'left' === wpbf_sidebar_layout() ) {
		get_sidebar();
	}

}
add_action( 'wpbf_sidebar_left', 'wpbf_do_sidebar_left' );

/**
 * Sidebar layout.
 *
 * @return string The sidebar layout.
 */
function wpbf_sidebar_layout() {

	// Set default sidebar position.
	$sidebar = get_theme_mod( 'sidebar_position', 'right' );

	$archive_sidebar_position = get_theme_mod( 'archive_sidebar_layout', 'global' );
	$sidebar                  = 'global' !== $archive_sidebar_position ? $archive_sidebar_position : $sidebar;

	if ( is_singular() && ! is_page() ) {

		$single_sidebar_position        = get_post_meta( get_the_ID(), 'wpbf_sidebar_position', true );
		$single_sidebar_position_global = get_theme_mod( 'single_sidebar_layout', 'global' );

		$sidebar = 'global' !== $single_sidebar_position_global ? $single_sidebar_position_global : $sidebar;
		$sidebar = $single_sidebar_position && 'global' !== $single_sidebar_position ? $single_sidebar_position : $sidebar;

	}

	if ( is_page() ) {

		$single_sidebar_position        = get_post_meta( get_the_ID(), 'wpbf_sidebar_position', true );
		$single_sidebar_position_global = get_theme_mod( 'single_sidebar_layout', 'global' );

		// By default there is no sidebar on pages.
		$sidebar = 'none';

		// Backwards compatibility. For pages that have the sidebar template selected, we inherit the global settings.
		if ( is_page_template( 'page-sidebar.php' ) ) {
			$sidebar = 'global' !== $single_sidebar_position_global ? $single_sidebar_position_global : $sidebar;
		}

		$sidebar = $single_sidebar_position && 'global' !== $single_sidebar_position ? $single_sidebar_position : $sidebar;

	}

	if ( is_404() ) {
		// There is no sidebar on 404 pages.
		// We do this to output the correct body class.
		$sidebar = 'none';
	}

	return apply_filters( 'wpbf_sidebar_layout', $sidebar );

}

/**
 * Article meta.
 *
 * Construct sortable article meta.
 */
function wpbf_article_meta() {

	$blog_meta = get_theme_mod( 'blog_sortable_meta', array( 'author', 'date' ) );

	if ( is_array( $blog_meta ) && ! empty( $blog_meta ) ) {

		do_action( 'wpbf_before_article_meta' );
		echo '<p class="article-meta">';
		do_action( 'wpbf_article_meta_open' );

		foreach ( $blog_meta as $value ) {

			switch ( $value ) {
				case 'author':
					do_action( 'wpbf_before_author_meta' );
					do_action( 'wpbf_author_meta' );
					do_action( 'wpbf_after_author_meta' );
					break;
				case 'date':
					do_action( 'wpbf_before_date_meta' );
					do_action( 'wpbf_date_meta' );
					do_action( 'wpbf_after_date_meta' );
					break;
				case 'comments':
					do_action( 'wpbf_before_comments_meta' );
					do_action( 'wpbf_comments_meta' );
					do_action( 'wpbf_after_comments_meta' );
					break;
				default:
					break;
			}
		}

		do_action( 'wpbf_article_meta_close' );
		echo '</p>';
		do_action( 'wpbf_after_article_meta' );

	}

}

/**
 * Article meta (author).
 */
function wpbf_do_author_meta() {

	$rtl         = is_rtl();
	$avatar      = get_theme_mod( 'blog_author_avatar' );
	$avatar_size = apply_filters( 'wpbf_author_meta_avatar_size', 128 );

	if ( ! $rtl && $avatar ) {
		echo get_avatar( get_the_author_meta( 'ID' ), $avatar_size );
	}

	echo sprintf(
		'<span class="article-author author vcard" itemscope="itemscope" itemprop="author" itemtype="https://schema.org/Person"><a class="url fn" href="%1$s" title="%2$s" rel="author" itemprop="url"><span itemprop="name">%3$s</span></a></span>',
		esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
		esc_attr( sprintf( __( 'View all posts by %s', 'page-builder-framework' ), get_the_author() ) ),
		esc_html( get_the_author() )
	);

	if ( $rtl && $avatar ) {
		echo get_avatar( get_the_author_meta( 'ID' ), $avatar_size );
	}

	echo '<span class="article-meta-separator">' . apply_filters( 'wpbf_article_meta_separator', ' | ' ) . '</span>';

}
add_action( 'wpbf_author_meta', 'wpbf_do_author_meta' );

/**
 * Article meta (date).
 */
function wpbf_do_date_meta() {

	echo '<span class="posted-on">' . __( 'Posted on', 'page-builder-framework' ) . '</span> <time class="article-time published" datetime="' . get_the_date( 'c' ) . '" itemprop="datePublished">' . get_the_date() . '</time>';
	echo '<span class="article-meta-separator">' . apply_filters( 'wpbf_article_meta_separator', ' | ' ) . '</span>';

}
add_action( 'wpbf_date_meta', 'wpbf_do_date_meta' );

/**
 * Article meta (comments).
 */
function wpbf_do_comments_meta() {

	echo '<span class="comments-count">';

	comments_number(
		__( '<span>No</span> Comments', 'page-builder-framework' ),
		__( '<span>1</span> Comment', 'page-builder-framework' ),
		__( '<span>%</span> Comments', 'page-builder-framework' )
	);

	echo '</span>';

	echo '<span class="article-meta-separator">' . apply_filters( 'wpbf_article_meta_separator', ' | ' ) . '</span>';

}
add_action( 'wpbf_comments_meta', 'wpbf_do_comments_meta' );

/**
 * Blog layout.
 *
 * @return array The blog layout.
 */
function wpbf_blog_layout() {

	$template_parts_header  = get_theme_mod( 'archive_sortable_header', array( 'title', 'meta', 'featured' ) );
	$template_parts_content = get_theme_mod( 'archive_sortable_content', array( 'excerpt' ) );
	$template_parts_footer  = get_theme_mod( 'archive_sortable_footer', array( 'readmore', 'categories' ) );
	$blog_layout            = get_theme_mod( 'archive_layout', 'default' );
	$style                  = get_theme_mod( 'archive_post_style', 'plain' );
	$stretched              = get_theme_mod( 'archive_boxed_image_streched', false );

	if ( 'beside' !== $blog_layout && 'boxed' === $style && $stretched ) {
		$style .= ' stretched';
	}

	return apply_filters(
		'wpbf_blog_layout',
		array(
			'blog_layout'            => $blog_layout,
			'template_parts_header'  => $template_parts_header,
			'template_parts_content' => $template_parts_content,
			'template_parts_footer'  => $template_parts_footer,
			'style'                  => $style,
		)
	);

}

/**
 * Single layout.
 *
 * @return array The single layout.
 */
function wpbf_post_layout() {

	$template_parts_header = get_theme_mod( 'single_sortable_header', array( 'title', 'meta', 'featured' ) );
	$template_parts_footer = get_theme_mod( 'single_sortable_footer', array( 'categories' ) );
	$post_layout           = 'default';
	$style                 = get_theme_mod( 'single_post_style', 'plain' );
	$style                .= get_theme_mod( 'single_boxed_image_stretched', false ) ? ' stretched' : '';

	return apply_filters(
		'wpbf_post_layout',
		array(
			'post_layout'           => $post_layout,
			'template_parts_header' => $template_parts_header,
			'template_parts_footer' => $template_parts_footer,
			'style'                 => $style,
		)
	);

}

/**
 * Declare menu's.
 *
 * Declare wp_nav_menu based on selected menu variation.
 */
function wpbf_nav_menu() {

	$custom_menu   = get_theme_mod( 'menu_custom' );
	$menu_position = get_theme_mod( 'menu_position' );

	if ( $custom_menu ) {

		echo do_shortcode( $custom_menu );

	} elseif ( in_array( $menu_position, array( 'menu-off-canvas', 'menu-off-canvas-left' ) ) ) {

		// Off canvas menu.
		wp_nav_menu(
			array(
				'theme_location' => 'main_menu',
				'container'      => false,
				'menu_class'     => 'wpbf-menu',
				'depth'          => 3,
				'fallback_cb'    => 'wpbf_main_menu_fallback',
			)
		);

	} elseif ( 'menu-full-screen' === $menu_position ) {

		// Full screen menu.
		wp_nav_menu(
			array(
				'theme_location' => 'main_menu',
				'container'      => false,
				'menu_class'     => 'wpbf-menu',
				'depth'          => 1,
				'fallback_cb'    => 'wpbf_main_menu_fallback',
			)
		);

	} elseif ( 'menu-vertical-left' === $menu_position ) {

		// Full screen menu.
		wp_nav_menu(
			array(
				'theme_location' => 'main_menu',
				'container'      => false,
				'menu_class'     => 'wpbf-menu',
				'depth'          => 1,
				'fallback_cb'    => 'wpbf_main_menu_fallback',
			)
		);

	} else {

		// Default menu.
		wp_nav_menu(
			array(
				'theme_location' => 'main_menu',
				'container'      => false,
				'menu_class'     => 'wpbf-menu wpbf-sub-menu' . wpbf_sub_menu_alignment() . wpbf_sub_menu_animation() . wpbf_menu_hover_effect(),
				'depth'          => 4,
				'fallback_cb'    => 'wpbf_main_menu_fallback',
			)
		);

	}

}
add_action( 'wpbf_main_menu', 'wpbf_nav_menu' );

/**
 * Declare mobile menu's.
 *
 * Declare wp_nav_menu based on selected mobile menu variation.
 */
function wpbf_mobile_nav_menu() {

	$custom_menu   = get_theme_mod( 'menu_custom' );
	$menu_position = get_theme_mod( 'mobile_menu_options', 'menu-mobile-hamburger' );

	if ( $custom_menu ) {

		echo do_shortcode( $custom_menu );

	} else {

		wp_nav_menu(
			array(
				'theme_location' => 'mobile_menu',
				'container'      => false,
				'menu_class'     => 'wpbf-mobile-menu',
				'depth'          => 4,
				'fallback_cb'    => 'wpbf_mobile_menu_fallback',
			)
		);

	}

}
add_action( 'wpbf_mobile_menu', 'wpbf_mobile_nav_menu' );

/**
 * Render main menu.
 *
 * Render main menu based on selected menu variation.
 */
function wpbf_menu() {
	get_template_part( 'inc/template-parts/navigation/' . apply_filters( 'wpbf_menu_variation', get_theme_mod( 'menu_position', 'menu-right' ) ) );
}
add_action( 'wpbf_navigation', 'wpbf_menu' );

/**
 * Render mobile menu.
 *
 * Render mobile menu based on selected mobile menu variation.
 */
function wpbf_mobile_menu() {
	get_template_part( 'inc/template-parts/navigation/' . apply_filters( 'wpbf_mobile_menu_variation', get_theme_mod( 'mobile_menu_options', 'menu-mobile-hamburger' ) ) );
}
add_action( 'wpbf_mobile_navigation', 'wpbf_mobile_menu' );

/**
 * Is off canvas menu check.
 *
 * Simple check to determine wether an off canvas menu is used.
 *
 * @return boolean.
 */
function wpbf_is_off_canvas_menu() {

	if ( in_array( get_theme_mod( 'menu_position' ), array( 'menu-off-canvas', 'menu-off-canvas-left', 'menu-full-screen' ) ) ) {
		return true;
	} else {
		return false;
	}

}

/**
 * Add sub menu indicators (SVG's) to regular menus.
 *
 * @param string  $title The title.
 * @param object  $item The menu item data object.
 * @param object  $args The arguments.
 * @param integer $depth Depth of menu item.
 *
 * @return string The updated nav menu item title.
 */
function wpbf_sub_menu_indicators( $title, $item, $args, $depth ) {

	// Let's stop if menu is not meant to have sub-menu's.
	if ( strpos( $args->menu_class, 'wpbf-sub-menu' ) === false ) {
		return $title;
	}

	// Add arrow icon if menu item has children.
	if ( isset( $item->classes ) && in_array( 'menu-item-has-children', $item->classes, true ) ) {

		if ( wpbf_svg_enabled() ) {
			$title .= ' ' . wpbf_svg( 'arrow-down' );
		} else {
			$title .= ' <i class="wpbff wpbff-arrow-down" aria-hidden="true"></i>';
		}

	}

	return $title;

}
add_filter( 'nav_menu_item_title', 'wpbf_sub_menu_indicators', 10, 4 );

/**
 * Add sub menu indicators to mobile & off canvas menu's.
 *
 * @param string  $item_output The menu item's starting HTML output.
 * @param object  $item The menu item data object.
 * @param integer $depth Depth of menu item.
 * @param object  $args The arguments.
 *
 * @return string The updated mobile menu item's starting HTML output.
 */
function wpbf_mobile_sub_menu_indicators( $item_output, $item, $depth, $args ) {

	if ( 'mobile_menu' === $args->theme_location || ( in_array( get_theme_mod( 'menu_position' ), array( 'menu-off-canvas', 'menu-off-canvas-left' ) ) && 'main_menu' === $args->theme_location ) ) {

		if ( isset( $item->classes ) && in_array( 'menu-item-has-children', $item->classes, true ) ) {

			if ( wpbf_svg_enabled() ) {

				$item_output .=
				'<button class="wpbf-submenu-toggle" aria-expanded="false">
					<span class="screen-reader-text">' . __( 'Menu Toggle', 'page-builder-framework' ) . '</span>
					' . wpbf_svg( 'arrow-down' ) . wpbf_svg( 'arrow-up' ) . '
				</button>';

			} else {

				$item_output .= '<button class="wpbf-submenu-toggle" aria-expanded="false"><span class="screen-reader-text">' . __( 'Menu Toggle', 'page-builder-framework' ) . '</span><i class="wpbff wpbff-arrow-down" aria-hidden="true"></i></button>';

			}

		}

	}

	return $item_output;

}
add_filter( 'walker_nav_menu_start_el', 'wpbf_mobile_sub_menu_indicators', 10, 4 );

/**
 * Submenu alignment class.
 *
 * @return string The submenu alignment class.
 */
function wpbf_sub_menu_alignment() {

	$sub_menu_alignment = get_theme_mod( 'sub_menu_alignment', 'left' );

	return ' wpbf-sub-menu-align-' . $sub_menu_alignment;

}

/**
 * Submenu animation class.
 *
 * @return string The submenu animation class.
 */
function wpbf_sub_menu_animation() {

	$sub_menu_animation = get_theme_mod( 'sub_menu_animation', 'fade' );

	return ' wpbf-sub-menu-animation-' . $sub_menu_animation;

}

/**
 * Menu alignment class.
 *
 * @return string The menu alignment class.
 */
function wpbf_menu_alignment() {

	$alignment = get_theme_mod( 'menu_alignment', 'left' );

	return ' menu-align-' . $alignment;

}

/**
 * Navigation hover effect classes.
 *
 * @return string The navigation hover effect classes.
 */
function wpbf_menu_hover_effect() {

	$menu_effect           = get_theme_mod( 'menu_effect', 'none' );
	$menu_effect_animation = get_theme_mod( 'menu_effect_animation', 'fade' );
	$menu_effect_alignment = get_theme_mod( 'menu_effect_alignment', 'center' );

	$hover_effect  = ' wpbf-menu-effect-' . $menu_effect;
	$hover_effect .= ' wpbf-menu-animation-' . $menu_effect_animation;
	$hover_effect .= ' wpbf-menu-align-' . $menu_effect_alignment;

	return $hover_effect;

}

/**
 * Navigation attributes.
 */
function wpbf_navigation_attributes() {

	$submenu_animation_duration = get_theme_mod( 'sub_menu_animation_duration' );
	$navigation_attributes      = $submenu_animation_duration ? 'data-sub-menu-animation-duration="' . esc_attr( $submenu_animation_duration ) . '"' : 'data-sub-menu-animation-duration="250"';

	echo apply_filters( 'wpbf_navigation_attributes', $navigation_attributes );

}

/**
 * Logo attributes.
 */
function wpbf_logo_attributes() {

	$attributes = '';

	return apply_filters( 'wpbf_logo_attributes', $attributes );

}

/**
 * Responsive embed/oembed.
 *
 * @param string $html The HTML output.
 * @param string $url The embed URL.
 * @param array  $attr Array of shortcode attributes.
 *
 * @return string The updated HTML output.
 */
function wpbf_responsive_embed( $html, $url, $attr ) {

	$providers = array( 'vimeo.com', 'youtube.com', 'youtu.be', 'wistia.com', 'wistia.net' );

	if ( wpbf_strposa( $url, $providers ) ) {
		$html = '<div class="wpbf-responsive-embed">' . $html . '</div>';
	}

	return $html;

}
add_filter( 'embed_oembed_html', 'wpbf_responsive_embed', 10, 3 );

/**
 * Page builder compatibility.
 *
 * Make the page full-width & remove the title if Page Builder is being used.
 *
 * @param int $id the ID.
 */
function wpbf_page_builder_compatibility( $id ) {

	// Stop here if we're not on a page.
	if ( 'page' !== get_post_type() ) {
		return;
	}

	$elementor  = get_post_meta( $id, '_elementor_edit_mode', true );
	$fl_enabled = get_post_meta( $id, '_fl_builder_enabled', true );

	if ( $fl_enabled || 'builder' === $elementor ) {

		$wpbf_stored_meta = get_post_meta( $id );

		$mydata = $wpbf_stored_meta['wpbf_options'];
		$mydata = is_array( $mydata ) ? $mydata : array();

		// Stop here if auto conversion already took place.
		if ( in_array( 'auto-convert', $mydata, true ) ) {
			return;
		}

		$mydata[] = 'remove-title';
		$mydata[] = 'full-width';
		$mydata[] = 'auto-convert';

		update_post_meta( $id, 'wpbf_options', $mydata );

	}

}
// add_action( 'wpbf_page_builder_compatibility', 'task' );

/**
 * Convert PHP array to JavaScript object.
 *
 * @param array $array Array to parse.
 * @return string
 */
function wpbf_array_to_js_object( $array ) {

	foreach ( $array as $key => $value ) {
		if ( ! is_scalar( $value ) ) {
			continue;
		}

		$array[ $key ] = html_entity_decode( (string) $value, ENT_QUOTES, 'UTF-8' );
	}

	return wp_json_encode( $array );

}
